using System.Collections;
using System.Collections.Generic;
using System;
using System.Runtime.InteropServices;
using System.Text;
using System.Globalization;
using Microsoft.Win32.SafeHandles;
using Unity.Collections.LowLevel.Unsafe;
using UnityEngine;

public static class ILSCONSTANTS
{

    public const int TESTNUMPOINTERS = 80;                              //only for testing to be able to reduce the number of pointers if Unity is too slow!!!
                                                                        //the game should use the value dwNumberOfPointersSupported in the struct ILS_PROPERTIES
                                                                        //so in a final game use this to be compatible once we increase the number of
                                                                        //pointers up to 600 e.g. in a planetarium or large cinema
    public const int MYTESTPOINTER = 26;                                //this pointer generates ILSPTRCOMMAND_VIBRATE_SHORT when the right mouse button is clicked
    public const int NUMCOMMANDS = 10;                                  //the number of command might be extended in the future

    //command codes used by the game
    public const int ILSPTRCOMMAND_VIBRATE_SHORT = 0x00;                //vibrate short
    public const int ILSPTRCOMMAND_VIBRATE_NORMAL = 0x01;               //vibrate normal
    public const int ILSPTRCOMMAND_VIBRATE_LONG = 0x02;                 //vibrate long
    public const int ILSPTRCOMMAND_VIBRATE_SHORT_DOUBLE = 0x03;         //vibrate short 2 times
    public const int ILSPTRCOMMAND_VIBRATE_NORMAL_DOUBLE = 0x04;        //vibrate normal 2 times
    public const int ILSPTRCOMMAND_VIBRATE_LONG_DOUBLE = 0x05;          //vibrate long 2 times
    public const int ILSPTRCOMMAND_LED_ON_500MSEC_RED = 0x06;           //turn-on LED red for 500msec
    public const int ILSPTRCOMMAND_LED_ON_500MSEC_YELLOW = 0x07;        //turn-on LED yellow for 500msec
    public const int ILSPTRCOMMAND_LED_ON_500MSEC_GREEN = 0x08;         //turn-on LED green for 500msec
    public const int ILSPTRCOMMAND_LASER_RED_ON_500MSEC = 0x09;         //turn-on the red laser for 500msec

    //command codes used by the camera calibration (corner detection) within script "CornerPoints.cs"
    public const int ILSCTRLCOMMAND_CORNER_DETECTION_START = 0x50;      //clear the current corners which turns-off the trapezoid correction (co-ordinates are garbage then)
    public const int ILSCTRLCOMMAND_CORNER_DETECTION_END = 0x51;        //turn-off master red laser
    public const int ILSCTRLCOMMAND_CORNER_DETECTION_RETRIGGER = 0x52;  //turn-on master red laser for 1000msec, must be sent every 800msec
    public const int ILSCTRLCOMMAND_CORNER_DETECTION_SET = 0x53;        //set the corners TLX,TLY,TRX,TRY,BLX,BLY,BRX,BRY

    public const int ILSMEMSTRUCTSIZE = 39380588;               //for checking, that the C++ side generates the same memory structure as we have here
    public const int ILSMEMSTRUCTVERSION = 0x22121401;          //for checking, that the C++ side generates the same memory structure as we have here

    public const int ILSMAXCAMS = 16;
    public const int ILSMAXPOINTERS = 600;
    public const int ILSFIFOSIZEXYK = 800 * 20;
    public const int ILSFIFOSIZECMDPTR = 800;
    public const int ILSFIFOSIZEPTRLOGIN = 800;
    public const int ILSFIFOSIZEPTRASSIGN = 800;

    public const int LOGLINECHARS = 256;
    public const int ILSFIFOSIZELOGPROC = 200;
    public const int ILSFIFOSIZELOGXYK = 200;
    public const int ILSFIFOSIZELOGPTRCMD = 200;
    public const int ILSFIFOSIZELOGLOGIN = 200;
    public const int ILSFIFOSIZELOGASSIGN = 200;
    public const int ILSFIFOSIZELOGGAME = 200;
}

public class SettingsFromMyINI 
{
    public static String strPathToSharedMemoryINI;

    //0 = do not show the magic circles, the pointers 0, MYTESTPOINTER and 79 generate commands to the pointers on left, right or both click 
    //1 = show the magic circles, all pointers generate commands to the pointers if the click inside one of the magic circles
    public static int iGameMode = 1;
    public static UInt32 dwMasterPointer = 0;
    public static bool bDebugLogSharedMemoryDetails = false;          //set to true if we shall output shared memory attach details
    public static float fRadius = 3.0f;
    public static float fMoveSpeed = 3.0f;    // speed of travel (seconds to make a complete round trip)
}

//***************************************************************************************************************************************************************************
//*                                                                                                                                                                         *
//* IGNORE IGNORE IGNORE IGNORE IGNORE IGNORE IGNORE IGNORE IGNORE IGNORE IGNORE IGNORE IGNORE IGNORE IGNORE IGNORE IGNORE IGNORE IGNORE IGNORE IGNORE IGNORE IGNORE IGNORE *
//*                                                                                                                                                                         *
//*  I would have placed this into an #include header file but C# does not support header files, so please ignore this part, it is needed to access the shared memory only  *
//*  ---------------------------------------------------------------------------------------------------------------------------------------------------------------------  *
//*                                                                                                                                                                         *
//***************************************************************************************************************************************************************************

//******************
//* ILS_PROPERTIES *
//******************

[StructLayout(LayoutKind.Sequential)]
unsafe struct ILS_PROPERTIES
{
    public UInt32 dwSizeOfStructure;                                    //for packing analysis with different compilers
    public UInt32 dwFeaturesSupported;                                  //features supported (t.b.d.)
    public UInt32 dwVersionCamWinUSBService;                            //version of CamWinUSBService 0xHHLLBB00 (high, low, build, revision=0x00)
    public UInt32 dwNumberOfCamerasSupported;                           //number of cameras supported (set to 1 in the moment)(t.b.d.)
    public UInt32 dwNumberOfPointersSupported;                          //number of pointer supported by the camera (set to 80 in the moment)(t.b.d.)
    public UInt32 dwNumberOfFramesPerSecond;			                //number of frames per second (set to 20 in the moment)(t.b.d.)
};

//**********************
//* ILS_COMMAND_STATUS *
//**********************

[StructLayout(LayoutKind.Sequential)]
unsafe struct ILS_COMMAND_STATUS
{
    public UInt32 dwSizeOfStructure;                                    //for packing analysis with different compilers
    public UInt32 dwCommand2Service;                                    //in the moment there is only 1 command: 0x000000FF Terminate (service sets it to 0 upon start)
    public UInt32 dwStatusService;                                      //0x00000001 service running, 0x80000000 service error
    public UInt32 dwServiceErrorCode;                                   //0x00000000 service writes the error code here if bit 31 of dwStatusService has been set
    public UInt32 dwStatusGame;                                         //in the moment we do not care, might be used for power saving, sleep, etc.
    public UInt32 dwServiceAliveCounter;				                //whenever the service has completed a data fetch from our devices, this number is incremented
};

//**********
//* ILSXYK *
//**********

[StructLayout(LayoutKind.Sequential)]
unsafe struct ILSXYK
{
    public UInt32 dwFrameNumber;                                        //cyclically 0...19, 0...19, etc.
    public UInt32 dwPointerNumber;                                      //cyclically 0...79, 0...79, etc.
    public UInt32 dwX;                                                  //X co-ordinate
    public UInt32 dwY;                                                  //Y co-ordinate
    public UInt32 dwKey;								                //key and position status
};

//****************************
//* ILS_FIFO_C2G_POINTER_XYK *
//****************************

[StructLayout(LayoutKind.Sequential)]
unsafe struct ILS_FIFO_C2G_POINTER_XYK
{
    public UInt32 dwSizeOfStructure;                                    //for packing analysis with different compilers
    public UInt32 dwWriteIndex;                                         //to be changed by CamWinUSBService only
    public UInt32 dwReadIndex;                                          //to be changed by the game only
    public UInt32 dwEntries;                                            //dummy member, we access the FIFO by offset ILS_FIFO_C2G_POINTER_XYK + 12
};

//*********************************
//* ILS_FIFO_G2C_POINTER_COMMANDS *
//*********************************

[StructLayout(LayoutKind.Sequential)]
unsafe struct ILSPCMD
{
    public UInt32 dwCommandCode;                                        //the command sent to the pointer(s)
    public fixed UInt32 dwPointerSelect[ILSCONSTANTS.ILSMAXPOINTERS];   //each UInt32 selects a pointer, if 0x00000000, a pointer is not selected, if 0xFFFFFFFF a pointer is selected
};

[StructLayout(LayoutKind.Sequential)]
unsafe struct ILS_FIFO_G2C_POINTER_COMMANDS
{
    public UInt32 dwSizeOfStructure;                                    //for packing analysis with different compilers
    public UInt32 dwWriteIndex;                                         //to be changed by the game only
    public UInt32 dwReadIndex;                                          //to be changed by CamWinUSBService only
    public UInt32 dwEntries;                                            //dummy member, we access the FIFO by offset ILS_FIFO_C2G_POINTER_XYK + 12
};

//******************
//* ILS_ALL_MEMORY *
//******************

[StructLayout(LayoutKind.Sequential)]
unsafe struct ILS_ALL_MEMORY
{
    public UInt32 dwSizeOfStructure;                                                    //for packing analysis with different compilers
    public UInt32 dwVersionOfStructure;                                                 //for packing analysis with different compilers
    public UInt32 dwOfsILS_PROPERTIES;                                                  //CILSMemory::OpenSharedMemory() sets the offset before allowing access to ILS_ALL_MEMORY
    public UInt32 dwOfsILS_COMMAND_STATUS;                                              //CILSMemory::OpenSharedMemory() sets the offset before allowing access to ILS_ALL_MEMORY
    public fixed UInt32 dwOfsILS_FIFO_C2G_POINTER_XYK[ILSCONSTANTS.ILSMAXCAMS];         //CILSMemory::OpenSharedMemory() sets the offset before allowing access to ILS_ALL_MEMORY
    public fixed UInt32 dwOfsILS_FIFO_G2C_POINTER_COMMANDS[ILSCONSTANTS.ILSMAXCAMS];    //CILSMemory::OpenSharedMemory() sets the offset before allowing access to ILS_ALL_MEMORY
    public fixed UInt32 dwOfsILS_FIFO_C2G_POINTER_LOGIN[ILSCONSTANTS.ILSMAXCAMS];       //CILSMemory::OpenSharedMemory() sets the offset before allowing access to ILS_ALL_MEMORY
    public fixed UInt32 dwOfsILS_FIFO_G2C_POINTER_ASSIGN[ILSCONSTANTS.ILSMAXCAMS];      //CILSMemory::OpenSharedMemory() sets the offset before allowing access to ILS_ALL_MEMORY
    public UInt32 dwOfsILS_FIFO_C2L_LOG_PROC;                                           //CILSMemory::OpenSharedMemory() sets the offset before allowing access to ILS_ALL_MEMORY
    public UInt32 dwOfsILS_FIFO_C2L_LOG_XYK;                                            //CILSMemory::OpenSharedMemory() sets the offset before allowing access to ILS_ALL_MEMORY
    public UInt32 dwOfsILS_FIFO_C2L_LOG_PTRCMD;                                         //CILSMemory::OpenSharedMemory() sets the offset before allowing access to ILS_ALL_MEMORY
    public UInt32 dwOfsILS_FIFO_C2L_LOG_LOGIN;                                          //CILSMemory::OpenSharedMemory() sets the offset before allowing access to ILS_ALL_MEMORY
    public UInt32 dwOfsILS_FIFO_C2L_LOG_ASSIGN;                                         //CILSMemory::OpenSharedMemory() sets the offset before allowing access to ILS_ALL_MEMORY
    public UInt32 dwOfsILS_FIFO_G2L_LOG_GAME;                                           //CILSMemory::OpenSharedMemory() sets the offset before allowing access to ILS_ALL_MEMORY
    public UInt32 dwOfsILS_FIFO_C2G_POINTER_XYK_CTRL;                                   //CILSMemory::OpenSharedMemory() sets the offset before allowing access to ILS_ALL_MEMORY
    public UInt32 dwOfsILS_FIFO_G2C_POINTER_COMMANDS_CTRL;		                        //CILSMemory::OpenSharedMemory() sets the offset before allowing access to ILS_ALL_MEMORY
    public fixed byte cSeparationCharStringZeroTerminated[4000];		                //CILSMemory::OpenSharedMemory() copies a test string there
};

//***************************************************************************************************************************************************************************
//*                                                                                                                                                                         *
//*  IGNORE END IGNORE END IGNORE END IGNORE END IGNORE END IGNORE END IGNORE END IGNORE END IGNORE END IGNORE END IGNORE END IGNORE END IGNORE END IGNORE END IGNORE END   *
//*                                                                                                                                                                         *
//*  I would have placed this into an #include header file but C# does not support header file, so please ignore this part, it is needed to access the shared memory only.  *
//*  ---------------------------------------------------------------------------------------------------------------------------------------------------------------------  *
//*                                                                                                                                                                         *
//*                                                                                   END                                                                                   *
//*                                                                                                                                                                         *
//***************************************************************************************************************************************************************************


unsafe public class ILSMemoryInterface : MonoBehaviour
{

    //***********************************************************************************************
    //* with these public variables ILSMemoryInterface allows to set the pointer coordinate scaling *
    //***********************************************************************************************
    //used for calculation of worldspace units from the ILS camera integers values 0...65535 for X and Y
    //
    //we make these variables public and do not use methods to get and set them for speed reasons!
    public float flTopLeftX;
    public float flTopLeftY;
    public float flBottomRightX;
    public float flBottomRightY;
    public float flWidth;
    public float flHeight;
    public float flDirX;
    public float flDirY;

    //**************************************************************************************************
    //* with these public variables ILSMemoryInterface allows other scripts to access the pointer data *
    //**************************************************************************************************
    //
    //we make these variables public and do not use methods to get and set them for speed reasons!
    public float[] flPointerX = new float[ILSCONSTANTS.ILSMAXPOINTERS];
    public float[] flPointerY = new float[ILSCONSTANTS.ILSMAXPOINTERS];
    public bool[] bPointerPosValid = new bool[ILSCONSTANTS.ILSMAXPOINTERS];
    public bool[] bPointerKeysValid = new bool[ILSCONSTANTS.ILSMAXPOINTERS];
    public bool[] bPointerButtonLPressed = new bool[ILSCONSTANTS.ILSMAXPOINTERS];
    public bool[] bPointerButtonRPressed = new bool[ILSCONSTANTS.ILSMAXPOINTERS];
    public bool[] bPointerButtonLClicked = new bool[ILSCONSTANTS.ILSMAXPOINTERS];
    public bool[] bPointerButtonRClicked = new bool[ILSCONSTANTS.ILSMAXPOINTERS];

    //*********************************************************************************************************
    //* with these public variables ILSMemoryInterface allows other scripts to access the master pointer data *
    //*********************************************************************************************************
    //
    //we make these variables public and do not use methods to get and set them for speed reasons!
    public float flPointerMasterX = 0;
    public float flPointerMasterY = 0;
    public UInt32 dwPointerMasterX = 0;
    public UInt32 dwPointerMasterY = 0;
    public bool bPointerMasterPosValid = false;
    public bool bPointerMasterKeysValid = false;
    public bool bPointerMasterButtonLPressed = false;
    public bool bPointerMasterButtonRPressed = false;
    public bool bPointerMasterButtonLClicked = false;
    public bool bPointerMasterButtonRClicked = false;
    public UInt32 dwPointerMasterCamera = 0;

    //*********************************************************************************************************
    //* with these public variables ILSMemoryInterface allows other scripts to write commands to the pointers *
    //*********************************************************************************************************
    //For every command x to pointer y, there is a dwCommandChanged[x] and a script must set this value to 1 so that function ILSMemoryInterface.UpdateCommands()
    //knows, that a dwCommandCollectorUInt32[x,y] has been changed and must be sent at the end of a frame. Each dwCommandCollectorUInt32[?,y] must be
    //set to 1 to select pointer y to execute the command. This allows to select 1, 10 or e.g. all pointers to execute the command by sending only
    //a single command and not e.g. 80 commands for 80 pointers.
    public uint[] dwCommandChanged = new UInt32[ILSCONSTANTS.NUMCOMMANDS];
    public uint[,] dwCommandCollector = new UInt32[ILSCONSTANTS.NUMCOMMANDS, ILSCONSTANTS.ILSMAXPOINTERS];

    //*******************************************************************************
    //* We use 5 Windows functions from 2 different DLLs, you can ignore this block *
    //*******************************************************************************
    [DllImport("kernel32.dll", SetLastError = true, CharSet = CharSet.Auto)]
    static extern SafeFileHandle OpenFileMapping(UInt32 dwDesiredAccess, bool bInheritHandle, String lpName);

    [DllImport("kernel32.dll", SetLastError = true)]
    static extern byte* MapViewOfFile(SafeFileHandle hFileMappingObject, UInt32 dwDesiredAccess, UInt32 dwFileOffsetHigh, UInt32 dwFileOffsetLow, UIntPtr pdwNumberOfBytesToMap);

    [DllImport("kernel32.dll", SetLastError = true)]
    static extern bool UnmapViewOfFile(byte* pucILS);

    [DllImport("kernel32.dll")]
    static extern int GetPrivateProfileString(String section, String key, String def, StringBuilder retVal, int size, string filePath);

    [System.Runtime.InteropServices.DllImport("user32.dll")]
    private static extern System.IntPtr GetActiveWindow();
    [DllImport("user32.dll", SetLastError = true)]
    static extern int MessageBox(IntPtr hwnd, String lpText, String lpCaption, uint uType);

    //privates
    const UInt32 STANDARD_RIGHTS_REQUIRED = 0x000F0000;
    const UInt32 SECTION_QUERY = 0x0001;
    const UInt32 SECTION_MAP_WRITE = 0x0002;
    const UInt32 SECTION_MAP_READ = 0x0004;
    const UInt32 SECTION_MAP_EXECUTE = 0x0008;
    const UInt32 SECTION_EXTEND_SIZE = 0x0010;
    const UInt32 SECTION_ALL_ACCESS = (STANDARD_RIGHTS_REQUIRED | SECTION_QUERY | SECTION_MAP_WRITE | SECTION_MAP_READ | SECTION_MAP_EXECUTE | SECTION_EXTEND_SIZE);
    const UInt32 FILE_MAP_ALL_ACCESS = SECTION_ALL_ACCESS;
    private SafeFileHandle sHandle;
    private IntPtr hHandle;

    //we must generate a set of pointers to access the unmanaged memory within the shared memory block.
    //The memory block is mapped into the address space of the game without C# nor the garbage collector (GC) knowing it
    //We receive a pointer to byte (byte*) from Windows function MapViewOfFile. We need the pointer to byte for calculating
    //the offsets of all other structures. Keep in mind: The real data is behind the struct ILS_ALL_MEMORY but we can not
    //define a complex struct with embedded structures with embedded arrays of structures because C# simply does not support
    //it. So the shared memory creator, either "ILSMemoryTest.EXE" (simulator) or "CamWinUSBService.EXE" (real camera interface)
    //writes all offsets of the structures behind ILS_ALL_MEMORY into all variables which start with "dwOfs", e.g. dwOfsILS_PROPERTIES
    //By adding the "dwOfs" varaibles to pucILSAllMemory (the byte* pointer!!!) we get the absolute addresses in the memory as
    //a pointer to byte (byte*). This we must cast to ILS_PROPERTIES* etc, also to the array of pointers ILS_FIFO_C2G_POINTER_XYK*[]
    //because they exist for every camera. After having done the casting for all pointers, we have access to the members accoring
    //to the above defined C# compatible mini structures.
    private byte* pucILSAllMemory;
    private ILS_ALL_MEMORY* pstILSAllMemory;
    private ILS_PROPERTIES* pstILSProperties;
    private ILS_COMMAND_STATUS* pstILSCommandStatus;
    private ILS_FIFO_C2G_POINTER_XYK*[] pstFifoC2GXYK = new ILS_FIFO_C2G_POINTER_XYK*[ILSCONSTANTS.ILSMAXCAMS];
    private ILS_FIFO_G2C_POINTER_COMMANDS*[] pstFifoG2CPointerCommand = new ILS_FIFO_G2C_POINTER_COMMANDS*[ILSCONSTANTS.ILSMAXCAMS];
    private ILS_FIFO_C2G_POINTER_XYK* pstFifoC2GXYKCtrl;
    private ILS_FIFO_G2C_POINTER_COMMANDS* pstFifoG2CPointerCommandCtrl;

    //These to pointers to structures we need to access the entries of the FIFOs. In C++ code, they defined as 
    //ILSXYK stEntriesXYK[xxx] but this is not possible in C# (no arrays of structures). So there is a dummy variable in each of
    //the FIFO structures named "dwEntries". We nevers use this member directly. It is only there to calculate the offset of the
    //entries within a outer structure. The offset of the outer structure we already know by the previous pointers. So we can use
    //"pstEntriesXYK = (ILSXYK*)&pstFifoC2GXYK[dwCamera]->dwEntries;" to generate the absolute address of the first (!!!) entry in the
    //FIFO. To access all entries, we simply use e.g. pstEntriesXYK[dwElement] and copy (memcpy) the data into a target managed
    //struct ILSXYK stEntry by calling e.g. FIFOXYK_Read(dwCamera, &stEntry);
    private ILSXYK* pstEntriesXYK;
    private ILSPCMD* pstEntriesPCMD;

    private bool bAttachSuccessful;
    private bool bUpdatePrintedSharedMemoryError;
    private string strErrorText;
    private bool bFIFOXYKFlushed = false;

    //these 2 variables we use to detect a click which is recognized by the last button state being false and
    //the new button state being true and always copy new button state to last button state after detection.
    private bool[] bPointerButtonLPressedLast = new bool[ILSCONSTANTS.ILSMAXPOINTERS];
    private bool[] bPointerButtonRPressedLast = new bool[ILSCONSTANTS.ILSMAXPOINTERS];
    private bool bPointerMasterButtonLPressedLast = false;
    private bool bPointerMasterButtonRPressedLast = false;


    // Start is called before the first frame update
    void Start()
    {
        UInt32 x;

        Debug.Log("ILSMemoryInterface.Start");

        //we must get the path to our !MYINI.INI file which is different within Unity Editor and at runtime in the build application
        //for both we ask for the assets folder
        String strPathToMyINI = Application.dataPath;
        //we can use the definition "UNITY_EDITOR" to find out if we run within Unity or standalone
        #if UNITY_EDITOR
            //ApplicationDataPath: C:/Users/Administrator/Documents/Unity/2D/ILSMemoryTest1/Assets
            strPathToMyINI += "/!MYINI.INI";
        #else
            //ApplicationDataPath: D:/TransferVMs/ILStec/ILS Software/Apps/ILSMemoryTest1/ILSMemoryTest1_Data
            int index = strPathToMyINI.LastIndexOf("/");
            strPathToMyINI = strPathToMyINI.Substring(0, index);
            strPathToMyINI += "/!MYINI.INI";
        #endif
        //in both cases there is no need to replace / slash by \ backslash, winodws can work with both
        StringBuilder temp = new StringBuilder(255);
        //read INI: [Paths], PathToSharedMemoryINI=
        //=========================================
        //default value is "NOTFOUND"
        GetPrivateProfileString("Paths", "PathToSharedMemoryINI", "NOTFOUND", temp, 255,strPathToMyINI);
        String strA = temp.ToString();
        //it must be found so detect "NOTFOUND"
        if (strA == "NOTFOUND")
        {
            WindowsMessageBox("Can not read file !MYINI.INI, [Paths], PathToSharedMemoryINI=");
            QuitTheGame();
        }
        SettingsFromMyINI.strPathToSharedMemoryINI = strA;

        //read INI: [Paths], PathToSharedMemoryINI=
        //=========================================
        GetPrivateProfileString("General", "DebugLogSharedMemoryDetails", "FALSE", temp, 255, strPathToMyINI);
        strA = temp.ToString();
        strA = strA.ToUpper();
        if (strA == "FALSE") SettingsFromMyINI.bDebugLogSharedMemoryDetails = false;          //set to true if we shall output shared memory attach details
        else                 SettingsFromMyINI.bDebugLogSharedMemoryDetails = true;

        //read INI: [Control], GameMode=
        //==============================
        GetPrivateProfileString("Control", "GameMode", "1", temp, 255, strPathToMyINI);
        strA = temp.ToString();
        SettingsFromMyINI.iGameMode = int.Parse(strA);
        
        //read INI: [Control], MasterPointer=
        //===================================
        GetPrivateProfileString("Control", "MasterPointer", "0", temp, 255, strPathToMyINI);
        strA = temp.ToString();
        SettingsFromMyINI.dwMasterPointer = (UInt32)(int.Parse(strA));

        //read INI: [Control], Radius=
        //============================
        GetPrivateProfileString("Control", "Radius", "3.0", temp, 255, strPathToMyINI);
        strA = temp.ToString();
        SettingsFromMyINI.fRadius = float.Parse(strA, new CultureInfo("en-US").NumberFormat);
        if (SettingsFromMyINI.fRadius < 1.0f)
        {
            WindowsMessageBox("[Control], Radius is too small (min. 1.0)");
            QuitTheGame();
        }

        //read INI: [Control], MoveSpeed=
        //===============================
        GetPrivateProfileString("Control", "MoveSpeed", "3.0", temp, 255, strPathToMyINI);
        strA = temp.ToString();
        SettingsFromMyINI.fMoveSpeed = float.Parse(strA, new CultureInfo("en-US").NumberFormat);
        if ((SettingsFromMyINI.fMoveSpeed > 100.0f)||(SettingsFromMyINI.fMoveSpeed < 0.2f))
        {
            WindowsMessageBox("[Control], MoveSpeed is out of range (0.2...10.0)");
            QuitTheGame();
        }

        //Attention     : This code for detecting the camera's viewport and calculating the correction multipliers to convert ILS camera
        //                coordinates to worldspace units only works for orthographic cameras. If you add a perspective or make any changes
        //                which influence the camera's viewport then it could be necessary to copy this code into the Update() function
        //                of this class so that it gets updated after any change
        //Proposal found: Use camera.pixelHeight and camera.pixelWidth and then ScreenPointToRay to get the 3D coordinates of all
        //                4 corners of the camera viewport at any given distance you might need.
        //                I did not test it!
        Camera camera = Camera.main;
        float flHalfHeight = camera.orthographicSize;
        float flHalfWidth = camera.aspect * flHalfHeight;
        flTopLeftX = -flHalfWidth;
        flTopLeftY = flHalfHeight;
        flWidth = flHalfWidth * 2f;
        flHeight = flHalfHeight*2f;
        flBottomRightX = flTopLeftX + flWidth;
        flBottomRightY = flTopLeftY - flHeight;
        flDirX = 1.0f;                                  //the ILS camera goes from left 0 to right 65535, so the direction is the same as in Unity
        flDirY = -1.0f;                                 //the ILS camera goes from top 0 to bottom 65535, so the direction is reverse compared to Unity
        Debug.Log("Camera TopLeft: " + flTopLeftX.ToString() + "/" + flTopLeftY.ToString() + " Camera BottomRight: " + flBottomRightX.ToString() + "/" + flBottomRightY.ToString());

        bUpdatePrintedSharedMemoryError = false;

        //temp = new StringBuilder(255); 
        GetPrivateProfileString("Memory", "SharedMemoryName", "NOTFOUND", temp, 255, SettingsFromMyINI.strPathToSharedMemoryINI);
        String strILSMemName = temp.ToString();
        if (strILSMemName=="NOTFOUND")
        {
            WindowsMessageBox("Can not open ILSMemory.INI, please check the pathes in file !MYINI.INI");
            QuitTheGame();
        }

        sHandle = new SafeFileHandle(hHandle, true);
        bAttachSuccessful = Attach(strILSMemName);
        if (!bAttachSuccessful)
        {
            WindowsMessageBox(strErrorText);
            QuitTheGame();
        }
        //initialize the pointer variables
        for (x = 0; x < ILSCONSTANTS.ILSMAXPOINTERS; x++)
        {
            flPointerX[x] = 0.0f;                      //all cursors in the center
            flPointerY[x] = 0.0f;                      //all cursors in the center
            bPointerPosValid[x] = false;               //all cursor positions are invalid
            bPointerKeysValid[x] = false;              //all cursor button states are invalid
            bPointerButtonLPressed[x] = false;         //all cursors no left button pressed
            bPointerButtonRPressed[x] = false;         //all cursors no right button pressed
            bPointerButtonLClicked[x] = false;         //all cursors no left button clicked
            bPointerButtonRClicked[x] = false;         //all cursors no right button clicked
        }
        //initialize the command collector
        for (x = 0; x < ILSCONSTANTS.NUMCOMMANDS; x++)
        {
            for (int y = 0; y < ILSCONSTANTS.ILSMAXPOINTERS; y++)
            {
                dwCommandCollector[x,y]=0;
            }
            dwCommandChanged[x] = 0;
        }
    }

    void WindowsMessageBox(string text)
    {
        MessageBox(GetActiveWindow(), text, "ILSMemoryInterface - Error", 0x00000040);
    }

    void QuitTheGame()
    {
        Debug.Log("ButtonHandler.QuitTheGame");
        //terminate the game in the built application mode (it does not work within the editor play mode)
        Application.Quit();
        //terminate the game in the editor play mode
        #if UNITY_EDITOR
            UnityEditor.EditorApplication.isPlaying = false;
        #endif
    }

    //this is the code to allow C#/Unity to access the unmanaged memory which has been mapped into the physical memory
    //of the game. You can disable all the "Debug.Log". I have added them only to be able to check if the C# side
    //generates the same absolute addresses as the C++ side. The C# side can also detect if everything worked well
    //if it checks the members "dwSizeOfStructure" which every of the above structures have a 1st member. But keep in
    //mind: C# get a mini verson of the real structures, so you can not compare with sizeof(struct), they will be different.
    //In the log file "ILSMemoryTest.LOG" you can find the real values. THey do not change until we decide to chage the
    //structure, but then ILSMEMSTRUCTVERSION will be diferent and you should immediately stop the game, because it might
    //happen that you get a NullPointerExecpetion or even worse, your game might corrupt other variables inside
    //the game (this happened to me during development) or outside the game which normally should be suppressed by
    //Windows (protected memory).
    unsafe private bool Attach(string SharedMemoryName)
    {
        int         x;          //general purpose integer for loops etc.
        UInt32      dwLeo;      //general purpose unsigned 32 bit integer to debug logs etc.

        Debug.Log("Entered ILSMemoryInterface.Attach");
        if (!sHandle.IsInvalid && !sHandle.IsClosed)
        {
            strErrorText = "Error: Can not open shared memory because it is already open! We close it now!";
            Debug.Log(strErrorText);
            sHandle.Close();
            return false;
        }
        Debug.Log("Opening shared memory");
        sHandle = OpenFileMapping(FILE_MAP_ALL_ACCESS, false, SharedMemoryName);
        if (sHandle.IsInvalid)
        {
            strErrorText = "Error: Can not open shared memory, was the service running at start of the game? You can not start the service once the game is started!!!";
            Debug.Log(strErrorText);
            return false;
        }
        Debug.Log("Shared memory opened successfully");
        //in a first step, I map only the first 256 bytes to get access to the total struct size and to the version of the struct
        if (SettingsFromMyINI.bDebugLogSharedMemoryDetails) Debug.Log("Mapping first 256 bytes of the Shared memory");
        pucILSAllMemory = MapViewOfFile(sHandle, FILE_MAP_ALL_ACCESS, 0, 0, new UIntPtr(256));
        if (pucILSAllMemory == (byte*)0)
        {
            strErrorText = "Error: Can not map first 256 byte of the shared memory, is the service running?";
            Debug.Log(strErrorText);
            sHandle.Close();
            return false;
        }
        if (SettingsFromMyINI.bDebugLogSharedMemoryDetails) Debug.Log("Generate pointers to the ILS_ALL_MEMORY struct");
        pstILSAllMemory = (ILS_ALL_MEMORY*)pucILSAllMemory;
        //get the size of the structure and the verson of the structure for checking them a few lines below
        UInt32 dwNumBytes = pstILSAllMemory->dwSizeOfStructure;
        UInt32 dwStructVersion = pstILSAllMemory->dwVersionOfStructure;
        //now I have the total size of the shared memory and the version, so I must unmap it now
        if (SettingsFromMyINI.bDebugLogSharedMemoryDetails) Debug.Log("Unmapping first 256 bytes of the Shared memory");
        if (!UnmapViewOfFile(pucILSAllMemory))
        {
            strErrorText = "UnmapViewOfFile returned an error but we continue with closing the file handle";
            Debug.Log(strErrorText);
            sHandle.Close();
            return false;
        }
        //check the size of the shared memory structure
        if (dwNumBytes != ILSCONSTANTS.ILSMEMSTRUCTSIZE)
        {
            strErrorText = "The structure size is not identical to what we expected!";
            Debug.Log(strErrorText);
            sHandle.Close();
            return false;
        }
        //check the version of the shared memory structure
        if (dwStructVersion != ILSCONSTANTS.ILSMEMSTRUCTVERSION)
        {
            strErrorText = "The structure version is not identical to what we expected!";
            Debug.Log(strErrorText);
            sHandle.Close();
            return false;
        }
        //in the second step, I map the whole memory area into this program's address space
        if (SettingsFromMyINI.bDebugLogSharedMemoryDetails) Debug.Log("Mapping all " + dwNumBytes.ToString() + " bytes of the Shared memory");
        UIntPtr puiTemp = new UIntPtr(dwNumBytes);
        pucILSAllMemory = MapViewOfFile(sHandle, FILE_MAP_ALL_ACCESS, 0, 0, puiTemp);
        if (pucILSAllMemory == (byte*)0)
        {
            strErrorText = "Error: Can not map full size shared memory, is the service running?";
            Debug.Log(strErrorText);
            sHandle.Close();
            return false;
        }
        //... and generate a pointer to the ILS_ALL_MEMORY structure
        pstILSAllMemory = (ILS_ALL_MEMORY*)pucILSAllMemory;
        dwLeo = (UInt32)pstILSAllMemory; if (SettingsFromMyINI.bDebugLogSharedMemoryDetails) Debug.Log("Absolute addess of pstILSAllMemory     : " + dwLeo.ToString());
        dwLeo = (UInt32)pstILSAllMemory->dwOfsILS_PROPERTIES;                     if (SettingsFromMyINI.bDebugLogSharedMemoryDetails) Debug.Log("Offset of ILS_PROPERTIES                   is " + dwLeo.ToString());
        dwLeo = (UInt32)pstILSAllMemory->dwOfsILS_COMMAND_STATUS;                 if (SettingsFromMyINI.bDebugLogSharedMemoryDetails) Debug.Log("Offset of ILS_COMMAND_STATUS               is " + dwLeo.ToString());
        for (x = 0; x < ILSCONSTANTS.ILSMAXCAMS; x++)
        {
            dwLeo = (UInt32)pstILSAllMemory->dwOfsILS_FIFO_C2G_POINTER_XYK[x]; if (SettingsFromMyINI.bDebugLogSharedMemoryDetails) Debug.Log("Offset of ILS_FIFO_C2G_POINTER_XYK[" + x.ToString() + "] is      " + dwLeo.ToString());
        }
        dwLeo = (UInt32)pstILSAllMemory->dwOfsILS_FIFO_C2G_POINTER_XYK_CTRL;      if (SettingsFromMyINI.bDebugLogSharedMemoryDetails) Debug.Log("Offset of FIFO_C2G_POINTER_XYK_CTRL        is " + dwLeo.ToString());
        dwLeo = (UInt32)pstILSAllMemory->dwOfsILS_FIFO_G2C_POINTER_COMMANDS_CTRL; if (SettingsFromMyINI.bDebugLogSharedMemoryDetails) Debug.Log("Offset of FIFO_G2C_POINTER_COMMANDS_CTRL   is " + dwLeo.ToString());

        if (SettingsFromMyINI.bDebugLogSharedMemoryDetails) Debug.Log("Generate pointers to the other embedded struct members");
        pstILSProperties = (ILS_PROPERTIES*)(pucILSAllMemory + pstILSAllMemory->dwOfsILS_PROPERTIES);
        dwLeo = (UInt32)pstILSProperties; if (SettingsFromMyINI.bDebugLogSharedMemoryDetails) Debug.Log("Absolute addess of pstILSProperties    : " + dwLeo.ToString());
        pstILSCommandStatus = (ILS_COMMAND_STATUS*)(pucILSAllMemory + pstILSAllMemory->dwOfsILS_COMMAND_STATUS);
        dwLeo = (UInt32)pstILSCommandStatus; if (SettingsFromMyINI.bDebugLogSharedMemoryDetails) Debug.Log("Absolute addess of pstILSCommandStatus : " + dwLeo.ToString());
        for (x = 0; x < ILSCONSTANTS.ILSMAXCAMS; x++)
        {
            pstFifoC2GXYK[x] = (ILS_FIFO_C2G_POINTER_XYK*)(pucILSAllMemory + pstILSAllMemory->dwOfsILS_FIFO_C2G_POINTER_XYK[x]);
            dwLeo = (UInt32)pstFifoC2GXYK[x]; if (SettingsFromMyINI.bDebugLogSharedMemoryDetails) Debug.Log("Absolute addess of pstFifoC2GXYK[" + x.ToString() + "]        : " + dwLeo.ToString());
            pstFifoG2CPointerCommand[x] = (ILS_FIFO_G2C_POINTER_COMMANDS*)(pucILSAllMemory + pstILSAllMemory->dwOfsILS_FIFO_G2C_POINTER_COMMANDS[x]);
            dwLeo = (UInt32)pstFifoG2CPointerCommand[x]; if (SettingsFromMyINI.bDebugLogSharedMemoryDetails) Debug.Log("Absolute addess of pstFifoG2CPointerCommand[" + x.ToString() + "]        : " + dwLeo.ToString());
        }
        pstFifoC2GXYKCtrl            = (ILS_FIFO_C2G_POINTER_XYK*)     (pucILSAllMemory + pstILSAllMemory->dwOfsILS_FIFO_C2G_POINTER_XYK_CTRL);
        dwLeo = (UInt32)pstFifoC2GXYKCtrl; if (SettingsFromMyINI.bDebugLogSharedMemoryDetails) Debug.Log("Absolute addess of pstFifoC2GXYKCtrl        : " + dwLeo.ToString());
        pstFifoG2CPointerCommandCtrl = (ILS_FIFO_G2C_POINTER_COMMANDS*)(pucILSAllMemory + pstILSAllMemory->dwOfsILS_FIFO_G2C_POINTER_COMMANDS_CTRL);
        dwLeo = (UInt32)pstFifoG2CPointerCommandCtrl; if (SettingsFromMyINI.bDebugLogSharedMemoryDetails) Debug.Log("Absolute addess of pstFifoG2CPointerCommandCtrl: " + dwLeo.ToString());
        Debug.Log("Final shared memory mapped sucessfully");
        UInt32 dwSizeOfStructure = pstILSAllMemory->dwSizeOfStructure;
        UInt32 dwVersionOfStructure = pstILSAllMemory->dwVersionOfStructure;
        if (SettingsFromMyINI.bDebugLogSharedMemoryDetails) Debug.Log("ILS_ALL_MEMORY.dwSizeOfStructure=" + dwSizeOfStructure.ToString() + " ILS_ALL_MEMORY.dwVersionOfStructure=" + dwVersionOfStructure.ToString("X8"));
        UInt32 ILS_PROPERTIESdwSizeOfStructure = pstILSProperties->dwSizeOfStructure;
        if (SettingsFromMyINI.bDebugLogSharedMemoryDetails) Debug.Log("ILS_PROPERTIES.dwSizeOfStructure           = " + ILS_PROPERTIESdwSizeOfStructure.ToString());
        UInt32 ILS_COMMAND_STATUSdwSizeOfStructure = pstILSCommandStatus->dwSizeOfStructure;
        if (SettingsFromMyINI.bDebugLogSharedMemoryDetails) Debug.Log("ILS_COMMAND_STATUS.dwSizeOfStructure       =  " + ILS_COMMAND_STATUSdwSizeOfStructure.ToString());
        for (x = 0; x < ILSCONSTANTS.ILSMAXCAMS; x++)
        {
            dwLeo = pstFifoC2GXYK[x]->dwSizeOfStructure;
            if (SettingsFromMyINI.bDebugLogSharedMemoryDetails) Debug.Log("ILS_FIFO_C2G_POINTER_XYK.dwSizeOfStructure =  " + dwLeo.ToString());
            dwLeo = pstFifoG2CPointerCommand[x]->dwSizeOfStructure;
            if (SettingsFromMyINI.bDebugLogSharedMemoryDetails) Debug.Log("ILS_FIFO_G2C_POINTER_COMMANDS.dwSizeOfStructure =  " + dwLeo.ToString());
        }
        UInt32 dwNumberOfCamerasSupported = pstILSProperties->dwNumberOfCamerasSupported;
        if (SettingsFromMyINI.bDebugLogSharedMemoryDetails) Debug.Log("dwNumberOfCamerasSupported = " + dwNumberOfCamerasSupported.ToString("X8"));
        String strConverted = ConvertStringZTCP1252toString(pstILSAllMemory->cSeparationCharStringZeroTerminated);
        if (SettingsFromMyINI.bDebugLogSharedMemoryDetails) Debug.Log("Conversion String result: " + strConverted);
        UInt32 dwCamera = 0;
        FIFOXYK_Flush(dwCamera);
        if (SettingsFromMyINI.bDebugLogSharedMemoryDetails)
        {
            dwLeo = pstFifoC2GXYK[0]->dwSizeOfStructure;              Debug.Log("pstFifoC2GXYK[0]->dwSizeOfStructure             =  " + dwLeo.ToString());
            dwLeo = pstFifoC2GXYK[0]->dwWriteIndex;                   Debug.Log("pstFifoC2GXYK[0]->dwWriteIndex                  =  " + dwLeo.ToString());
            dwLeo = pstFifoC2GXYK[0]->dwReadIndex;                    Debug.Log("pstFifoC2GXYK[0]->dwReadIndex                   =  " + dwLeo.ToString());
            dwLeo = pstFifoG2CPointerCommand[0]->dwSizeOfStructure;   Debug.Log("pstFifoG2CPointerCommand[0]->dwSizeOfStructure  =  " + dwLeo.ToString());
            dwLeo = pstFifoG2CPointerCommand[0]->dwWriteIndex;        Debug.Log("pstFifoG2CPointerCommand[0]->dwWriteIndex       =  " + dwLeo.ToString());
            dwLeo = pstFifoG2CPointerCommand[0]->dwReadIndex;         Debug.Log("pstFifoG2CPointerCommand[0]->dwReadIndex        =  " + dwLeo.ToString());
            dwLeo = pstFifoC2GXYKCtrl->dwSizeOfStructure;             Debug.Log("pstFifoC2GXYKCtrl->dwSizeOfStructure            =  " + dwLeo.ToString());
            dwLeo = pstFifoC2GXYKCtrl->dwWriteIndex;                  Debug.Log("pstFifoC2GXYKCtrl->dwWriteIndex                 =  " + dwLeo.ToString());
            dwLeo = pstFifoC2GXYKCtrl->dwReadIndex;                   Debug.Log("pstFifoC2GXYKCtrl->dwReadIndex                  =  " + dwLeo.ToString());
            dwLeo = pstFifoG2CPointerCommandCtrl->dwSizeOfStructure;  Debug.Log("pstFifoG2CPointerCommandCtrl->dwSizeOfStructure =  " + dwLeo.ToString());
            dwLeo = pstFifoG2CPointerCommandCtrl->dwWriteIndex;       Debug.Log("pstFifoG2CPointerCommandCtrl->dwWriteIndex      =  " + dwLeo.ToString());
            dwLeo = pstFifoG2CPointerCommandCtrl->dwReadIndex;        Debug.Log("pstFifoG2CPointerCommandCtrl->dwReadIndex       =  " + dwLeo.ToString());
        }
        return true;
    }
    
    //convert a string zero terminated from the C++ side (code page 1252) to a manged C# string
    unsafe public String ConvertStringZTCP1252toString(byte* pucStringZT)
    {
        String s = "";
        char[] bbb = new char[1];
        Encoding encoding = Encoding.GetEncoding(1252);
        //the max. lgenth of the source string is 4000 characters including the terminating zero
        for (int x = 0; x < 4000; x++)
        {
            bbb[0] = (char)*pucStringZT++;
            byte[] encBytes = encoding.GetBytes(bbb);
            byte[] unitcodeBytes = Encoding.Convert(encoding, Encoding.Unicode, encBytes);
            String ss = Encoding.Unicode.GetString(unitcodeBytes);
            if (bbb[0] == 0) break;
            s += ss;
        }
        return s;
    }

    unsafe private void Detach()
    {
        Debug.Log("ILSMemoryInterface.Detach");
        if (!UnmapViewOfFile(pucILSAllMemory))
        {
            Debug.Log("UnmapViewOfFile returned an error but we continue with closing the file handle");
        }
        if (!sHandle.IsInvalid && !sHandle.IsClosed)
        {
            Debug.Log("Closing the file handle");
            sHandle.Close();
        }
        else
        {
            Debug.Log("The file handle is invalid or it is already closed, we do not care!");
        }
        Debug.Log("Leaving ILSMemoryInterface.Detach");
    }

    void OnApplicationQuit()
    {
        Debug.Log("ILSMemoryInterface.OnApplicationQuit");
        if (bAttachSuccessful)
        {
            Detach();
        }
        Debug.Log("Leaving ILSMemoryInterface.OnApplicationQuit");
    }

    // Update is called once per frame
    unsafe void Update()
    {
        if (sHandle.IsInvalid)
        {
            if (bUpdatePrintedSharedMemoryError == false)
            {
                Debug.Log("Error: Can not access shared memory, was the service running at start? You can not start the service once the game is started!!!");
                bUpdatePrintedSharedMemoryError = true;
            }
            return;
        }
        //Debug.Log("ILSMemoryInterface.Update");

        UInt32 dwCamera = 0;
        if (bFIFOXYKFlushed == false)
        {
            FIFOXYK_Flush(dwCamera);
            bFIFOXYKFlushed = true;
        }
        //while FIFO is not empty
        while (!FIFOXYK_IsEmpty(dwCamera))
        {
            ILSXYK stEntry;
            FIFOXYK_Read(dwCamera, &stEntry);
            if (stEntry.dwPointerNumber < ILSCONSTANTS.TESTNUMPOINTERS)
            {
                //if the pointer position is valid
                //we declare the position valid only if the position (0x00008000) and the key status (0x00005000) is valid!!!
                if ((stEntry.dwKey & 0x0000C000) == 0x0000C000)
                {
                    //if bPointerPosValid is true, then the cursor might be displayed (visible) by the game, if it is false, it should not be visible!
                    //bPointerPosValid is false, if e.g. the pointer is outside the screen or the player logged out of the game or the pointer is changed to stand-by
                    bPointerPosValid[stEntry.dwPointerNumber] = true;
                    flPointerX[stEntry.dwPointerNumber] = flTopLeftX + flDirX * ((float)(stEntry.dwX)) * flWidth / 65536;
                    flPointerY[stEntry.dwPointerNumber] = flTopLeftY + flDirY * ((float)(stEntry.dwY)) * flHeight / 65536;
                }
                else
                {
                    bPointerPosValid[stEntry.dwPointerNumber] = false;
                }

                //if the pointer button state is valid
                //we only change the button state and button clicks if the keys are valid
                if ((stEntry.dwKey & 0x00004000) != 0)
                {
                    bPointerKeysValid[stEntry.dwPointerNumber] = true;
                    if ((stEntry.dwKey & 0x00000100) != 0) bPointerButtonLPressed[stEntry.dwPointerNumber] = true;
                    else                                   bPointerButtonLPressed[stEntry.dwPointerNumber] = false;

                    if ((stEntry.dwKey & 0x00000200) != 0) bPointerButtonRPressed[stEntry.dwPointerNumber] = true;
                    else                                   bPointerButtonRPressed[stEntry.dwPointerNumber] = false;

                    //detect a click when the L button state changed from false to true
                    if (
                        (bPointerButtonLPressed[stEntry.dwPointerNumber]==true)&&
                        (bPointerButtonLPressedLast[stEntry.dwPointerNumber]==false)
                       )
                    {
                        //the cursor handler must clear this flag to false once it recognized it
                        bPointerButtonLClicked[stEntry.dwPointerNumber] = true;
                    }
                    bPointerButtonLPressedLast[stEntry.dwPointerNumber] = bPointerButtonLPressed[stEntry.dwPointerNumber];

                    //detect a click when the R button state changed from false to true
                    if (
                        (bPointerButtonRPressed[stEntry.dwPointerNumber] == true) &&
                        (bPointerButtonRPressedLast[stEntry.dwPointerNumber] == false)
                       )
                    {
                        //the cursor handler must clear this flag to false once it recognized it
                        bPointerButtonRClicked[stEntry.dwPointerNumber] = true;
                    }
                    bPointerButtonRPressedLast[stEntry.dwPointerNumber] = bPointerButtonRPressed[stEntry.dwPointerNumber];
                }
                else
                {
                    bPointerKeysValid[stEntry.dwPointerNumber] = false;
                }
            }
        }

        //************************
        //** only in GameMode 2 **
        //************************
        //read the control FIFO where the master pointer entries arrive from every camera in the system
        //we read and update the MasterPointer variables only if the GameMode is 2 which is the corner detection
        //this can also be used if the Unity game is a control program (the only one!!!) for other games.
        //But it must be guaranteed that only one application (a Unity game or a C++ program) uses FIFOXYKCTRL
        //because if multiple programs read the FIFO the data gets inconsistent because the read pointers are updated
        //by diferent programs!!!

        //while GameMode==2 and FIFOXYKCTRL is not empty
        while ((SettingsFromMyINI.iGameMode == 2)&&(!FIFOXYKCTRL_IsEmpty()))
        {
            ILSXYK stEntry;
            FIFOXYKCTRL_Read(&stEntry);
            //the camera number is encoded into bits 8 to 15 of the frame number which can only be 0...19. The camera number is added 0xA0 before the entry is written into the FIFO
            dwPointerMasterCamera = ((stEntry.dwFrameNumber >> 8) & 0xFF) - 0xA0;
            //mask out the frame number (without the camera number)
            stEntry.dwFrameNumber &= 0xFF;
            //Debug.Log("Received Master Pointer - dwCamera=" + dwPointerMasterCamera.ToString() + " dwPointerNumber=" + stEntry.dwPointerNumber.ToString() + " X=" + stEntry.dwX.ToString() + " Y=" + stEntry.dwY.ToString());
            if (stEntry.dwPointerNumber == SettingsFromMyINI.dwMasterPointer)
            {
                //we must handle the master popinter in the same way as the other pointers but write the result not into an array but into "Master" decorated variables
                //if the pointer position is valid
                //we declare the position valid only if the position (0x00008000) and the key status (0x00005000) is valid!!!
                if ((stEntry.dwKey & 0x0000C000) == 0x0000C000)
                {
                    //if bPointerPosValid is true, then the cursor might be displayed (visible) by the game, if it is false, it should not be visible!
                    //bPointerPosValid is false, if e.g. the pointer is outside the screen or the player logged out of the game or the pointer is changed to stand-by
                    bPointerMasterPosValid = true;
                    flPointerMasterX = flTopLeftX + flDirX * ((float)(stEntry.dwX)) * flWidth / 65536;
                    flPointerMasterY = flTopLeftY + flDirY * ((float)(stEntry.dwY)) * flHeight / 65536;
                    dwPointerMasterX = stEntry.dwX;         //for calibration we need the orignal values and not the values scaled to the camera view
                    dwPointerMasterY = stEntry.dwY;         //for calibration we need the orignal values and not the values scaled to the camera view
                }
                else
                {
                    bPointerMasterPosValid = false;
                }

                //if the pointer button state is valid
                //we only change the button state and button clicks if the keys are valid
                if ((stEntry.dwKey & 0x00004000) != 0)
                {
                    bPointerMasterKeysValid = true;
                    if ((stEntry.dwKey & 0x00000100) != 0) bPointerMasterButtonLPressed = true;
                    else bPointerMasterButtonLPressed = false;

                    if ((stEntry.dwKey & 0x00000200) != 0) bPointerMasterButtonRPressed = true;
                    else bPointerMasterButtonRPressed = false;

                    //detect a click when the L button state changed from false to true
                    if (
                        (bPointerMasterButtonLPressed == true) &&
                        (bPointerMasterButtonLPressedLast == false)
                       )
                    {
                        //the cursor handler must clear this flag to false once it recognized it
                        bPointerMasterButtonLClicked = true;
                    }
                    bPointerMasterButtonLPressedLast = bPointerMasterButtonLPressed;

                    //detect a click when the R button state changed from false to true
                    if (
                        (bPointerMasterButtonRPressed == true) &&
                        (bPointerMasterButtonRPressedLast == false)
                       )
                    {
                        //the cursor handler must clear this flag to false once it recognized it
                        bPointerMasterButtonRClicked = true;
                    }
                    bPointerMasterButtonRPressedLast = bPointerMasterButtonRPressed;
                }
                else
                {
                    bPointerMasterKeysValid = false;
                }
            }
            //if it is camera 0 and it is the master pointer then we send a ILSCTRLCOMMAND_CORNER_DETECTION_RETRIGGER for testing
            if ((dwPointerMasterCamera == 0xA0)&&(stEntry.dwPointerNumber== SettingsFromMyINI.dwMasterPointer))
            {
                ILSPCMD stEntryCommand;
                stEntryCommand.dwCommandCode = ILSCONSTANTS.ILSCTRLCOMMAND_CORNER_DETECTION_RETRIGGER;
                FIFOPCMDSCTRL_Write(&stEntryCommand);
            }
        }
    }

    unsafe public bool SendControlCommand(UInt32 dwCmd, UInt32[] dwArgs)
    {
        ILSPCMD stEntry;
        int x;
        int iNumArgs = -1;
        if (dwCmd == ILSCONSTANTS.ILSCTRLCOMMAND_CORNER_DETECTION_START    ) iNumArgs = 0;      //dwArgs not used
        if (dwCmd == ILSCONSTANTS.ILSCTRLCOMMAND_CORNER_DETECTION_END      ) iNumArgs = 0;      //dwArgs not used
        if (dwCmd == ILSCONSTANTS.ILSCTRLCOMMAND_CORNER_DETECTION_RETRIGGER) iNumArgs = 0;      //dwArgs not used
        if (dwCmd == ILSCONSTANTS.ILSCTRLCOMMAND_CORNER_DETECTION_SET      ) iNumArgs = 8;      //dwArgs[0...7]
        if (iNumArgs == -1) return false;
        stEntry.dwCommandCode = dwCmd;
        for (x = 0;x < iNumArgs; x++)
        {
            stEntry.dwPointerSelect[x] = dwArgs[x];
        }
        return FIFOPCMDSCTRL_Write(&stEntry);
    }

    // Update is called once at the end of a frame by ILSMemoryUpdater.Update()
    unsafe public void UpdateCommands()
    {
        //Debug.Log("ILSMemoryInterface.UpdateCommands");
        //after all magic circles have been checked, we must clear all other click events
        for (UInt32 y = 0; y < ILSCONSTANTS.TESTNUMPOINTERS; y++)
        {
            bPointerButtonLClicked[y] = false;
            bPointerButtonRClicked[y] = false;
        }
        if (sHandle.IsInvalid)
        {
            if (bUpdatePrintedSharedMemoryError == false)
            {
                Debug.Log("Error: Can not access shared memory, was the service running at start? You can not start the service once the game is started!!!");
                bUpdatePrintedSharedMemoryError = true;
            }
            return;
        }
        UInt32 dwCamera = 0;
        if (FIFOPCMDS_IsFull(dwCamera))
        {
            Debug.Log("ILSMemoryInterface.UpdateCommands: ILS_FIFO_G2C_POINTER_COMMANDS is full");
            return;
        }
        for (UInt32 x = 0; x<ILSCONSTANTS.NUMCOMMANDS; x++)
        {
            if (dwCommandChanged[x] != 0)
            {
                //Debug.Log("ILSMemoryInterface.UpdateCommands: dwCommandChanged[" + x.ToString() + "] != 0");
                ILSPCMD stEntry;                                                                            //must be on the stack otherwise it is not fixed (see below Attention:)
                //I add a dummy value so that I can check it on the C++ side and a command code of 0x00 (potential empty entry) is avoided
                //ILSPTRCOMMAND_VIBRATE_SHORT           = 0x00    --> 0xA0
                //ILSPTRCOMMAND_VIBRATE_NORMAL          = 0x01    --> 0xA1
                //ILSPTRCOMMAND_VIBRATE_LONG            = 0x02    --> 0xA2
                //ILSPTRCOMMAND_VIBRATE_SHORT_DOUBLE    = 0x03    --> 0xA3
                //ILSPTRCOMMAND_VIBRATE_NORMAL_DOUBLE   = 0x04    --> 0xA4
                //ILSPTRCOMMAND_VIBRATE_LONG_DOUBLE     = 0x05    --> 0xA5
                //ILSPTRCOMMAND_LED_ON_500MSEC_RED      = 0x06    --> 0xA6
                //ILSPTRCOMMAND_LED_ON_500MSEC_YELLOW   = 0x07    --> 0xA7
                //ILSPTRCOMMAND_LED_ON_500MSEC_GREEN    = 0x08    --> 0xA8
                //ILSPTRCOMMAND_LASER_RED_ON_500MSEC    = 0x09    --> 0xA9
                stEntry.dwCommandCode = x + 0xA0;
                //copy the contents of the C# movable array "dwCommandCollector[x,ILSCONSTANTS.ILSMAXPOINTERS]" into the struct stEntry
                //can do this with a UnsafeUtility.MemCpy():
                //Attention: Using pointers to variables is very dangerous in C#/Unity because the garbage collector may relocate variables
                //           at any time to defragment the heap memory. Byte using the "fixed" statement, all variables for which pointers
                //           are declared in the arguments of the "Fixed" statement (in this case "p" points to dwCommandCollector[x,0])
                //           do not move for the time the statements within the curly braces are executed.
                //           MemCpy is much faster than copying ILSMAXPOINTERS variables to other variables because it copies the whole
                //           memory contents to the new memory location
                //           We do not have to take care of stEntry, because "A local variable in an unsafe method or a parameter is already fixed (on the stack)"
                //           Sorry folks, I can not use inefficient code, I am used to work with pointers
                fixed (UInt32* p = &dwCommandCollector[x,0])
                {
                    UnsafeUtility.MemCpy(&stEntry.dwPointerSelect[0], p, ILSCONSTANTS.ILSMAXPOINTERS * sizeof(UInt32));
                }
                FIFOPCMDS_Write(dwCamera,&stEntry);
                //set dwCommandChanged[x] to 0 to disable sending the same packet until a new pointer generated this command again
                dwCommandChanged[x] = 0;
                //also clear all UInt32 of dwCommandCollector[x, 0...ILSMAXPOINTERS-1]
                //can do this with a UnsafeUtility.MemSet():
                //Attention: Using pointers to variables is very dangerous in C#/Unity because the garbage collector may relocate variables
                //           at any time to defragment the heap memory. Byte using the "fixed" statement, all variables for which pointers
                //           are declared in the arguments of the "Fixed" statement (in this case "p" points to dwCommandCollector[x,0])
                //           do not move for the time the statements within the curly braces are executed.
                //           MemSet is much faster than setting ILSMAXPOINTERS variables to specific values
                //           Sorry folks, I can not use inefficient code, I am used to work with pointers
                fixed (UInt32* p = &dwCommandCollector[x, 0])
                {
                    UnsafeUtility.MemSet(p, 0, ILSCONSTANTS.ILSMAXPOINTERS * sizeof(UInt32));
                }
            }
        }
    }

    //***************************************************************************************************************************************************
    //***************************************************************************************************************************************************
    //**                                                                                                                                               **
    //**                                                           The FIFO access functions                                                           **
    //**                                                                                                                                               **
    //***************************************************************************************************************************************************
    //***************************************************************************************************************************************************

    //****************************
    //*                          *
    //* ILS_FIFO_C2G_POINTER_XYK *
    //*                          *
    //****************************

    unsafe void FIFOXYK_Flush(UInt32 dwCamera)
    {
        //in case of a C2G FIFO flushing means to set the dwReadIndex = dwWriteIndex, because we are not allowed to modify dwWriteIndex
        pstFifoC2GXYK[dwCamera]->dwReadIndex = pstFifoC2GXYK[dwCamera]->dwWriteIndex;
        return;
    }

    unsafe bool FIFOXYK_IsEmpty(UInt32 dwCamera)
    {
        if (pstFifoC2GXYK[dwCamera]->dwReadIndex == pstFifoC2GXYK[dwCamera]->dwWriteIndex) return true;
        else                                                                               return false;
    }

    unsafe bool FIFOXYK_IsFull(UInt32 dwCamera)
    {
        if (pstFifoC2GXYK[dwCamera]->dwWriteIndex < pstFifoC2GXYK[dwCamera]->dwReadIndex)
        {
            if (pstFifoC2GXYK[dwCamera]->dwWriteIndex == pstFifoC2GXYK[dwCamera]->dwReadIndex - 1) return true;
            else                                                                                   return false;
        }
        else
        {
            if (pstFifoC2GXYK[dwCamera]->dwWriteIndex == pstFifoC2GXYK[dwCamera]->dwReadIndex + ILSCONSTANTS.ILSFIFOSIZEXYK - 1) return true;
            else                                                                                                                 return false;
        }
    }

    unsafe UInt32 FIFOXYK_GetNumCurrentEntries(UInt32 dwCamera)
    {
        if (pstFifoC2GXYK[dwCamera]->dwWriteIndex < pstFifoC2GXYK[dwCamera]->dwReadIndex) { return ILSCONSTANTS.ILSFIFOSIZEXYK + pstFifoC2GXYK[dwCamera]->dwWriteIndex - pstFifoC2GXYK[dwCamera]->dwReadIndex; }
        else                                                                              { return pstFifoC2GXYK[dwCamera]->dwWriteIndex - pstFifoC2GXYK[dwCamera]->dwReadIndex; }
    }

    unsafe UInt32 FIFOXYK_GetNumFreeEntries(UInt32 dwCamera)
    {
        if (pstFifoC2GXYK[dwCamera]->dwWriteIndex < pstFifoC2GXYK[dwCamera]->dwReadIndex) { return pstFifoC2GXYK[dwCamera]->dwReadIndex - pstFifoC2GXYK[dwCamera]->dwWriteIndex - 1; }
        else                                                                              { return ILSCONSTANTS.ILSFIFOSIZEXYK + pstFifoC2GXYK[dwCamera]->dwReadIndex - pstFifoC2GXYK[dwCamera]->dwWriteIndex - 1; }
    }

    unsafe bool FIFOXYK_Read(UInt32 dwCamera, ILSXYK* pstEntry)
    {
        if (FIFOXYK_IsEmpty(dwCamera)) return false;
        //read entry
        //we first must get the absolute address of the array of ILSXYK entries (C# does not support an array of structs)
        pstEntriesXYK = (ILSXYK*)&pstFifoC2GXYK[dwCamera]->dwEntries;
        //then we use a MemCpy to copy the data from the source struct ILSXYK in the FIFO to the given destination struct ILSXYK (arg 2)
        UnsafeUtility.MemCpy(pstEntry, &pstEntriesXYK[pstFifoC2GXYK[dwCamera]->dwReadIndex], sizeof(ILSXYK));
        //increment read index
        if (pstFifoC2GXYK[dwCamera]->dwReadIndex >= ILSCONSTANTS.ILSFIFOSIZEXYK - 1) { pstFifoC2GXYK[dwCamera]->dwReadIndex = 0; }
        else                                                                         { pstFifoC2GXYK[dwCamera]->dwReadIndex++; }
        return true;
    }

    //*********************************
    //*                               *
    //* ILS_FIFO_G2C_POINTER_COMMANDS *
    //*                               *
    //*********************************

    unsafe void FIFOPCMDS_Flush(UInt32 dwCamera)
    {
        //in case of a G2C FIFO flushing means to set the dwWriteIndex = dwReadIndex, because we are not allowed to modify dwReadIndex
        pstFifoG2CPointerCommand[dwCamera]->dwWriteIndex = pstFifoG2CPointerCommand[dwCamera]->dwReadIndex;
        return;
    }

    unsafe bool FIFOPCMDS_IsEmpty(UInt32 dwCamera)
    {
        if (pstFifoG2CPointerCommand[dwCamera]->dwReadIndex == pstFifoG2CPointerCommand[dwCamera]->dwWriteIndex) return true;
        else                                                                                                     return false;
    }

    unsafe bool FIFOPCMDS_IsFull(UInt32 dwCamera)
    {
        if (pstFifoG2CPointerCommand[dwCamera]->dwWriteIndex < pstFifoG2CPointerCommand[dwCamera]->dwReadIndex)
        {
            if (pstFifoG2CPointerCommand[dwCamera]->dwWriteIndex == pstFifoG2CPointerCommand[dwCamera]->dwReadIndex - 1) return true;
            else                                                                                                         return false;
        }
        else
        {
            if (pstFifoG2CPointerCommand[dwCamera]->dwWriteIndex == pstFifoG2CPointerCommand[dwCamera]->dwReadIndex + ILSCONSTANTS.ILSFIFOSIZECMDPTR - 1) return true;
            else                                                                                                                                          return false;
        }
    }

    unsafe UInt32 FIFOPCMDS_GetNumCurrentEntries(UInt32 dwCamera)
    {
        if (pstFifoG2CPointerCommand[dwCamera]->dwWriteIndex < pstFifoG2CPointerCommand[dwCamera]->dwReadIndex) { return ILSCONSTANTS.ILSFIFOSIZECMDPTR + pstFifoG2CPointerCommand[dwCamera]->dwWriteIndex - pstFifoG2CPointerCommand[dwCamera]->dwReadIndex; }
        else                                                                                                    { return pstFifoG2CPointerCommand[dwCamera]->dwWriteIndex - pstFifoG2CPointerCommand[dwCamera]->dwReadIndex; }
    }

    unsafe UInt32 FIFOPCMDS_GetNumFreeEntries(UInt32 dwCamera)
    {
        if (pstFifoG2CPointerCommand[dwCamera]->dwWriteIndex < pstFifoG2CPointerCommand[dwCamera]->dwReadIndex) { return pstFifoG2CPointerCommand[dwCamera]->dwReadIndex - pstFifoG2CPointerCommand[dwCamera]->dwWriteIndex - 1; }
        else                                                                                                    { return ILSCONSTANTS.ILSFIFOSIZECMDPTR + pstFifoG2CPointerCommand[dwCamera]->dwReadIndex - pstFifoG2CPointerCommand[dwCamera]->dwWriteIndex - 1; }
    }

    unsafe bool FIFOPCMDS_Write(UInt32 dwCamera, ILSPCMD* pstEntry)
    {
        if (FIFOPCMDS_IsFull(dwCamera)) return false;
        //write entry
        //we first must get the absolute address of the array of ILSPCMD entries (C# does not support an array of structs)
        pstEntriesPCMD = (ILSPCMD*)&pstFifoG2CPointerCommand[dwCamera]->dwEntries;
        //then we use a MemCpy to copy the data from the given source struct ILSPCMD (arg 2) to the destination struct ILSPCMD in the FIFO
        UnsafeUtility.MemCpy(&pstEntriesPCMD[pstFifoG2CPointerCommand[dwCamera]->dwWriteIndex], pstEntry, sizeof(ILSPCMD));
        //increment write index
        if (pstFifoG2CPointerCommand[dwCamera]->dwWriteIndex >= ILSCONSTANTS.ILSFIFOSIZECMDPTR - 1) pstFifoG2CPointerCommand[dwCamera]->dwWriteIndex = 0;
        else                                                                                        pstFifoG2CPointerCommand[dwCamera]->dwWriteIndex++;
        return true;
    }

    //*********************************
    //*                               *
    //* ILS_FIFO_C2G_POINTER_XYK_CTRL *
    //*                               *
    //*********************************

    unsafe void FIFOXYKCTRL_Flush()
    {
        //in case of a C2G FIFO flushing means to set the dwReadIndex = dwWriteIndex, because we are not allowed to modify dwWriteIndex
        pstFifoC2GXYKCtrl->dwReadIndex = pstFifoC2GXYKCtrl->dwWriteIndex;
        return;
    }

    unsafe bool FIFOXYKCTRL_IsEmpty()
    {
        if (pstFifoC2GXYKCtrl->dwReadIndex == pstFifoC2GXYKCtrl->dwWriteIndex) return true;
        else                                                                   return false;
    }

    unsafe bool FIFOXYKCTRL_IsFull()
    {
        if (pstFifoC2GXYKCtrl->dwWriteIndex < pstFifoC2GXYKCtrl->dwReadIndex)
        {
            if (pstFifoC2GXYKCtrl->dwWriteIndex == pstFifoC2GXYKCtrl->dwReadIndex - 1) return true;
            else                                                                       return false;
        }
        else
        {
            if (pstFifoC2GXYKCtrl->dwWriteIndex == pstFifoC2GXYKCtrl->dwReadIndex + ILSCONSTANTS.ILSFIFOSIZEXYK - 1) return true;
            else                                                                                                     return false;
        }
    }

    unsafe UInt32 FIFOXYKCTRL_GetNumCurrentEntries()
    {
        if (pstFifoC2GXYKCtrl->dwWriteIndex < pstFifoC2GXYKCtrl->dwReadIndex) { return ILSCONSTANTS.ILSFIFOSIZEXYK + pstFifoC2GXYKCtrl->dwWriteIndex - pstFifoC2GXYKCtrl->dwReadIndex; }
        else                                                                  { return pstFifoC2GXYKCtrl->dwWriteIndex - pstFifoC2GXYKCtrl->dwReadIndex; }
    }

    unsafe UInt32 FIFOXYKCTRL_GetNumFreeEntries()
    {
        if (pstFifoC2GXYKCtrl->dwWriteIndex < pstFifoC2GXYKCtrl->dwReadIndex) { return pstFifoC2GXYKCtrl->dwReadIndex - pstFifoC2GXYKCtrl->dwWriteIndex - 1; }
        else                                                                  { return ILSCONSTANTS.ILSFIFOSIZEXYK + pstFifoC2GXYKCtrl->dwReadIndex - pstFifoC2GXYKCtrl->dwWriteIndex - 1; }
    }

    unsafe bool FIFOXYKCTRL_Read(ILSXYK* pstEntry)
    {
        if (FIFOXYKCTRL_IsEmpty()) return false;
        //read entry
        //we first must get the absolute address of the array of ILSXYK entries (C# does not support an array of structs)
        pstEntriesXYK = (ILSXYK*)&pstFifoC2GXYKCtrl->dwEntries;
        //then we use a MemCpy to copy the data from the source struct ILSXYK in the FIFO to the given destination struct ILSXYK (arg 2)
        UnsafeUtility.MemCpy(pstEntry, &pstEntriesXYK[pstFifoC2GXYKCtrl->dwReadIndex], sizeof(ILSXYK));
        //increment read index
        if (pstFifoC2GXYKCtrl->dwReadIndex >= ILSCONSTANTS.ILSFIFOSIZEXYK - 1) { pstFifoC2GXYKCtrl->dwReadIndex = 0; }
        else                                                                   { pstFifoC2GXYKCtrl->dwReadIndex++; }
        return true;
    }

    //**************************************
    //*                                    *
    //* ILS_FIFO_G2C_POINTER_COMMANDS_CTRL *
    //*                                    *
    //**************************************

    unsafe void FIFOPCMDSCTRL_Flush()
    {
        //in case of a G2C FIFO flushing means to set the dwWriteIndex = dwReadIndex, because we are not allowed to modify dwReadIndex
        pstFifoG2CPointerCommandCtrl->dwWriteIndex = pstFifoG2CPointerCommandCtrl->dwReadIndex;
        return;
    }

    unsafe bool FIFOPCMDSCTRL_IsEmpty()
    {
        if (pstFifoG2CPointerCommandCtrl->dwReadIndex == pstFifoG2CPointerCommandCtrl->dwWriteIndex) return true;
        else                                                                                         return false;
    }

    unsafe bool FIFOPCMDSCTRL_IsFull()
    {
        if (pstFifoG2CPointerCommandCtrl->dwWriteIndex < pstFifoG2CPointerCommandCtrl->dwReadIndex)
        {
            if (pstFifoG2CPointerCommandCtrl->dwWriteIndex == pstFifoG2CPointerCommandCtrl->dwReadIndex - 1) return true;
            else                                                                                             return false;
        }
        else
        {
            if (pstFifoG2CPointerCommandCtrl->dwWriteIndex == pstFifoG2CPointerCommandCtrl->dwReadIndex + ILSCONSTANTS.ILSFIFOSIZECMDPTR - 1) return true;
            else                                                                                                                              return false;
        }
    }

    unsafe UInt32 FIFOPCMDSCTRL_GetNumCurrentEntries()
    {
        if (pstFifoG2CPointerCommandCtrl->dwWriteIndex < pstFifoG2CPointerCommandCtrl->dwReadIndex) { return ILSCONSTANTS.ILSFIFOSIZECMDPTR + pstFifoG2CPointerCommandCtrl->dwWriteIndex - pstFifoG2CPointerCommandCtrl->dwReadIndex; }
        else                                                                                        { return pstFifoG2CPointerCommandCtrl->dwWriteIndex - pstFifoG2CPointerCommandCtrl->dwReadIndex; }
    }

    unsafe UInt32 FIFOPCMDSCTRL_GetNumFreeEntries()
    {
        if (pstFifoG2CPointerCommandCtrl->dwWriteIndex < pstFifoG2CPointerCommandCtrl->dwReadIndex) { return pstFifoG2CPointerCommandCtrl->dwReadIndex - pstFifoG2CPointerCommandCtrl->dwWriteIndex - 1; }
        else                                                                                        { return ILSCONSTANTS.ILSFIFOSIZECMDPTR + pstFifoG2CPointerCommandCtrl->dwReadIndex - pstFifoG2CPointerCommandCtrl->dwWriteIndex - 1; }
    }

    unsafe bool FIFOPCMDSCTRL_Write(ILSPCMD* pstEntry)
    {
        if (FIFOPCMDSCTRL_IsFull()) return false;
        //write entry
        //we first must get the absolute address of the array of ILSPCMD entries (C# does not support an array of structs)
        pstEntriesPCMD = (ILSPCMD*)&pstFifoG2CPointerCommandCtrl->dwEntries;
        //then we use a MemCpy to copy the data from the given source struct ILSPCMD (arg 2) to the destination struct ILSPCMD in the FIFO
        UnsafeUtility.MemCpy(&pstEntriesPCMD[pstFifoG2CPointerCommandCtrl->dwWriteIndex], pstEntry, sizeof(ILSPCMD));
        //increment write index
        if (pstFifoG2CPointerCommandCtrl->dwWriteIndex >= ILSCONSTANTS.ILSFIFOSIZECMDPTR - 1) pstFifoG2CPointerCommandCtrl->dwWriteIndex = 0;
        else pstFifoG2CPointerCommandCtrl->dwWriteIndex++;
        return true;
    }
}
